<?php
/**
 * Plugin Name: Tap and Try - Virtual Product Visualization
 * Plugin URI: https://tapandtry.com
 * Description: Let customers visualize your products in their space using AR/camera before buying. Increase conversions with interactive product try-on.
 * Version: 1.0.0
 * Author: Tap and Try
 * Author URI: https://tapandtry.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: tap-and-try
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 9.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('TNT_VERSION', '1.0.0');
define('TNT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('TNT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('TNT_API_URL', 'https://tapandtry.com/tapandtry-apps-api');
define('TNT_WIDGET_URL', 'https://tapandtry.com/widget.js');

// Declare WooCommerce HPOS compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

/**
 * Check if WooCommerce is active
 */
function tnt_is_woocommerce_active() {
    return class_exists('WooCommerce');
}

/**
 * Main Plugin Class
 */
class TapAndTry {
    
    private static $instance = null;
    private $api_key = null;
    
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function __construct() {
        $this->api_key = get_option('tnt_api_key', '');
        
        // Admin hooks (always load)
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'admin_scripts']);
        add_action('admin_notices', [$this, 'admin_notices']);
        
        // AJAX handlers (always load for dashboard links)
        add_action('wp_ajax_tnt_get_usage', [$this, 'ajax_get_usage']);
        add_action('wp_ajax_tnt_get_login_url', [$this, 'ajax_get_login_url']);
        
        // WooCommerce-dependent hooks
        if (tnt_is_woocommerce_active()) {
            // Frontend hooks
            add_action('wp_enqueue_scripts', [$this, 'frontend_scripts']);
            add_action('woocommerce_single_product_summary', [$this, 'add_try_button'], 25);
            
            // WooCommerce AJAX handlers
            add_action('wp_ajax_tnt_sync_products', [$this, 'ajax_sync_products']);
            add_action('wp_ajax_tnt_toggle_product', [$this, 'ajax_toggle_product']);
            
            // Product meta box
            add_action('add_meta_boxes', [$this, 'add_product_meta_box']);
            add_action('save_post_product', [$this, 'save_product_meta']);
        }
        
        // Shortcode (always available)
        add_shortcode('tapandtry', [$this, 'shortcode_handler']);
    }
    
    /**
     * Admin Notices
     */
    public function admin_notices() {
        // Only show on Tap and Try pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'tap-and-try') === false) {
            return;
        }
        
        if (!tnt_is_woocommerce_active()) {
            ?>
            <div class="notice notice-warning">
                <p><strong><?php _e('Tap and Try:', 'tap-and-try'); ?></strong> 
                <?php _e('WooCommerce is not installed or activated. Product sync features require WooCommerce. You can still access your Tap & Try dashboard and manage products at tapandtry.com.', 'tap-and-try'); ?></p>
            </div>
            <?php
        }
    }
    
    /**
     * Admin Menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Tap and Try', 'tap-and-try'),
            __('Tap and Try', 'tap-and-try'),
            'manage_options',
            'tap-and-try',
            [$this, 'render_admin_page'],
            'dashicons-camera',
            56
        );
        
        add_submenu_page(
            'tap-and-try',
            __('Products', 'tap-and-try'),
            __('Products', 'tap-and-try'),
            'manage_options',
            'tap-and-try-products',
            [$this, 'render_products_page']
        );
        
        add_submenu_page(
            'tap-and-try',
            __('Settings', 'tap-and-try'),
            __('Settings', 'tap-and-try'),
            'manage_options',
            'tap-and-try-settings',
            [$this, 'render_settings_page']
        );
    }
    
    /**
     * Register Settings
     */
    public function register_settings() {
        register_setting('tnt_settings', 'tnt_api_key', [
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        register_setting('tnt_settings', 'tnt_button_text', [
            'default' => 'Tap & Try It',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        register_setting('tnt_settings', 'tnt_checkout_type', [
            'default' => 'product_page',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        register_setting('tnt_settings', 'tnt_button_position', [
            'default' => 'after_add_to_cart',
            'sanitize_callback' => 'sanitize_text_field'
        ]);
        register_setting('tnt_settings', 'tnt_auto_sync', [
            'default' => false,
            'sanitize_callback' => 'rest_sanitize_boolean'
        ]);
        register_setting('tnt_settings', 'tnt_marketplace_visible', [
            'default' => false,
            'sanitize_callback' => 'rest_sanitize_boolean'
        ]);
    }
    
    /**
     * Admin Scripts
     */
    public function admin_scripts($hook) {
        if (strpos($hook, 'tap-and-try') === false) {
            return;
        }
        
        wp_enqueue_style('tnt-admin', TNT_PLUGIN_URL . 'assets/css/admin.css', [], TNT_VERSION);
        wp_enqueue_script('tnt-admin', TNT_PLUGIN_URL . 'assets/js/admin.js', ['jquery'], TNT_VERSION, true);
        
        wp_localize_script('tnt-admin', 'tntAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('tnt_admin_nonce'),
            'apiKey' => $this->api_key,
            'strings' => [
                'syncing' => __('Syncing...', 'tap-and-try'),
                'syncComplete' => __('Sync complete!', 'tap-and-try'),
                'syncError' => __('Sync failed. Please try again.', 'tap-and-try'),
            ]
        ]);
    }
    
    /**
     * Frontend Scripts
     */
    public function frontend_scripts() {
        if (!is_product()) {
            return;
        }
        
        global $post;
        $enabled = get_post_meta($post->ID, '_tnt_enabled', true);
        
        if ($enabled !== 'yes') {
            return;
        }
        
        // Load widget script
        wp_enqueue_script('tnt-widget', TNT_WIDGET_URL, [], TNT_VERSION, true);
        
        // Custom styles
        wp_add_inline_style('woocommerce-general', '
            .tnt-try-button {
                margin: 10px 0;
            }
        ');
    }
    
    /**
     * Add Try Button to Product Page
     */
    public function add_try_button() {
        global $post;
        
        $enabled = get_post_meta($post->ID, '_tnt_enabled', true);
        $tnt_product_id = get_post_meta($post->ID, '_tnt_product_id', true);
        $tnt_slug = get_post_meta($post->ID, '_tnt_slug', true);
        
        if ($enabled !== 'yes' || empty($tnt_product_id)) {
            return;
        }
        
        $button_text = get_option('tnt_button_text', 'Tap & Try');
        $identifier = !empty($tnt_slug) ? $tnt_slug : $tnt_product_id;
        
        echo '<div class="tnt-try-button" data-tnt-product="' . esc_attr($identifier) . '" data-tnt-text="' . esc_attr($button_text) . '"></div>';
    }
    
    /**
     * Product Meta Box
     */
    public function add_product_meta_box() {
        add_meta_box(
            'tnt_product_settings',
            __('Tap and Try', 'tap-and-try'),
            [$this, 'render_product_meta_box'],
            'product',
            'side',
            'default'
        );
    }
    
    public function render_product_meta_box($post) {
        wp_nonce_field('tnt_product_meta', 'tnt_product_nonce');
        
        $enabled = get_post_meta($post->ID, '_tnt_enabled', true);
        $tnt_product_id = get_post_meta($post->ID, '_tnt_product_id', true);
        $tnt_slug = get_post_meta($post->ID, '_tnt_slug', true);
        $marketplace = get_post_meta($post->ID, '_tnt_marketplace', true);
        
        ?>
        <p>
            <label>
                <input type="checkbox" name="tnt_enabled" value="yes" <?php checked($enabled, 'yes'); ?>>
                <?php _e('Enable Tap and Try', 'tap-and-try'); ?>
            </label>
        </p>
        
        <?php if (!empty($tnt_product_id)) : ?>
            <p>
                <strong><?php _e('Synced:', 'tap-and-try'); ?></strong><br>
                <small>ID: <?php echo esc_html($tnt_product_id); ?></small>
            </p>
        <?php else : ?>
            <p>
                <em><?php _e('Not synced yet. Save to sync.', 'tap-and-try'); ?></em>
            </p>
        <?php endif; ?>
        
        <p>
            <label>
                <input type="checkbox" name="tnt_marketplace" value="yes" <?php checked($marketplace, 'yes'); ?>>
                <?php _e('Show in Tap and Try Marketplace', 'tap-and-try'); ?>
            </label>
        </p>
        <?php
    }
    
    public function save_product_meta($post_id) {
        if (!isset($_POST['tnt_product_nonce']) || !wp_verify_nonce($_POST['tnt_product_nonce'], 'tnt_product_meta')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        $enabled = isset($_POST['tnt_enabled']) ? 'yes' : 'no';
        $marketplace = isset($_POST['tnt_marketplace']) ? 'yes' : 'no';
        
        update_post_meta($post_id, '_tnt_enabled', $enabled);
        update_post_meta($post_id, '_tnt_marketplace', $marketplace);
        
        // Sync to Tap and Try if enabled
        if ($enabled === 'yes' && !empty($this->api_key)) {
            $this->sync_single_product($post_id);
        }
    }
    
    /**
     * Sync Single Product to Tap and Try
     */
    private function sync_single_product($post_id) {
        $product = wc_get_product($post_id);
        if (!$product) {
            return false;
        }
        
        $tnt_product_id = get_post_meta($post_id, '_tnt_product_id', true);
        $marketplace = get_post_meta($post_id, '_tnt_marketplace', true) === 'yes';
        
        // Get checkout URL based on setting
        $checkout_url = $this->get_checkout_url($product);
        
        // Get pricing
        $regular_price = $product->get_regular_price();
        $sale_price = $product->get_sale_price();
        $currency = get_woocommerce_currency();
        
        $data = [
            'name' => $product->get_name(),
            'description' => $product->get_short_description(),
            'categories' => $this->get_product_categories($product),
            'externalId' => (string) $post_id,
            'externalUrl' => get_permalink($post_id),
            'checkoutUrl' => $checkout_url,
            'regularPrice' => $regular_price ? floatval($regular_price) : null,
            'specialPrice' => $sale_price ? floatval($sale_price) : null,
            'currency' => $currency,
            'buttonText' => 'Buy Now',  // Modal action button - not the product page button
            'marketplaceVisible' => $marketplace,
            'status' => $marketplace ? 'published' : 'draft',
            'sourcePlatform' => 'wordpress',
            'variants' => $this->get_product_variants($product, $checkout_url),
        ];
        
        $endpoint = !empty($tnt_product_id) 
            ? '/v1/plugin/products/' . $tnt_product_id 
            : '/v1/plugin/products';
        
        $method = !empty($tnt_product_id) ? 'PUT' : 'POST';
        
        $response = $this->api_request($endpoint, $method, $data);
        
        if ($response && isset($response['data']['product'])) {
            update_post_meta($post_id, '_tnt_product_id', $response['data']['product']['id']);
            update_post_meta($post_id, '_tnt_slug', $response['data']['product']['slug']);
            return true;
        }
        
        return false;
    }
    
    /**
     * Get Checkout URL based on setting
     */
    private function get_checkout_url($product) {
        $checkout_type = get_option('tnt_checkout_type', 'product_page');
        
        switch ($checkout_type) {
            case 'add_to_cart':
                // Direct add to cart URL (for simple products)
                if ($product->is_type('simple')) {
                    return add_query_arg('add-to-cart', $product->get_id(), wc_get_cart_url());
                }
                // Variable products need to go to product page to select options
                return get_permalink($product->get_id());
                
            case 'product_page':
            default:
                return get_permalink($product->get_id());
        }
    }
    
    /**
     * Get Product Variants (for variable products) or single variant (for simple products)
     */
    private function get_product_variants($product, $default_checkout_url) {
        $variants = [];
        $currency = get_woocommerce_currency();
        
        if ($product->is_type('variable')) {
            // Variable product - get all variations
            $variations = $product->get_available_variations();
            
            foreach ($variations as $index => $variation) {
                $variation_obj = wc_get_product($variation['variation_id']);
                if (!$variation_obj) continue;
                
                // Build variant name from attributes
                $attributes = [];
                foreach ($variation['attributes'] as $attr_key => $attr_value) {
                    if (!empty($attr_value)) {
                        $attributes[] = $attr_value;
                    }
                }
                $variant_name = !empty($attributes) ? implode(' / ', $attributes) : $product->get_name();
                
                // Get image URL
                $image_id = $variation_obj->get_image_id();
                $image_url = $image_id ? wp_get_attachment_url($image_id) : null;
                
                // Fallback to parent product image
                if (!$image_url) {
                    $parent_image_id = $product->get_image_id();
                    $image_url = $parent_image_id ? wp_get_attachment_url($parent_image_id) : null;
                }
                
                // Get variation-specific checkout URL
                $checkout_url = add_query_arg([
                    'add-to-cart' => $product->get_id(),
                    'variation_id' => $variation['variation_id'],
                ] + $variation['attributes'], wc_get_cart_url());
                
                $variants[] = [
                    'name' => $variant_name,
                    'imageUrl' => $image_url,
                    'regularPrice' => $variation_obj->get_regular_price() ? floatval($variation_obj->get_regular_price()) : null,
                    'specialPrice' => $variation_obj->get_sale_price() ? floatval($variation_obj->get_sale_price()) : null,
                    'currency' => $currency,
                    'checkoutUrl' => $checkout_url,
                    'infoUrl' => get_permalink($product->get_id()),
                    'externalId' => (string) $variation['variation_id'],
                    'sortOrder' => $index,
                ];
            }
        } else {
            // Simple product - create single variant from product images
            $image_ids = $product->get_gallery_image_ids();
            $main_image_id = $product->get_image_id();
            
            // Add main image first
            if ($main_image_id) {
                array_unshift($image_ids, $main_image_id);
            }
            
            // Remove duplicates
            $image_ids = array_unique($image_ids);
            
            foreach ($image_ids as $index => $image_id) {
                $image_url = wp_get_attachment_url($image_id);
                if (!$image_url) continue;
                
                $variants[] = [
                    'name' => $index === 0 ? $product->get_name() : $product->get_name() . ' - Image ' . ($index + 1),
                    'imageUrl' => $image_url,
                    'regularPrice' => $product->get_regular_price() ? floatval($product->get_regular_price()) : null,
                    'specialPrice' => $product->get_sale_price() ? floatval($product->get_sale_price()) : null,
                    'currency' => $currency,
                    'checkoutUrl' => $default_checkout_url,
                    'infoUrl' => get_permalink($product->get_id()),
                    'externalId' => (string) $product->get_id() . '-' . $index,
                    'sortOrder' => $index,
                ];
            }
        }
        
        return $variants;
    }
    
    /**
     * Get Product Categories
     */
    private function get_product_categories($product) {
        $categories = [];
        $terms = get_the_terms($product->get_id(), 'product_cat');
        
        if ($terms && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[] = $term->name;
            }
        }
        
        return $categories;
    }
    
    /**
     * API Request Helper
     */
    private function api_request($endpoint, $method = 'GET', $data = null) {
        if (empty($this->api_key)) {
            return false;
        }
        
        $args = [
            'method' => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ],
            'timeout' => 30,
        ];
        
        if ($data && in_array($method, ['POST', 'PUT', 'PATCH'])) {
            $args['body'] = json_encode($data);
        }
        
        $response = wp_remote_request(TNT_API_URL . $endpoint, $args);
        
        if (is_wp_error($response)) {
            error_log('Tap and Try API Error: ' . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }
    
    /**
     * AJAX: Sync Products
     */
    public function ajax_sync_products() {
        check_ajax_referer('tnt_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $products = wc_get_products([
            'limit' => -1,
            'meta_key' => '_tnt_enabled',
            'meta_value' => 'yes',
        ]);
        
        $synced = 0;
        $errors = [];
        
        foreach ($products as $product) {
            if ($this->sync_single_product($product->get_id())) {
                $synced++;
            } else {
                $errors[] = $product->get_name();
            }
        }
        
        wp_send_json_success([
            'synced' => $synced,
            'errors' => $errors,
            'message' => sprintf(__('%d products synced successfully.', 'tap-and-try'), $synced)
        ]);
    }
    
    /**
     * AJAX: Toggle Product
     */
    public function ajax_toggle_product() {
        check_ajax_referer('tnt_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $product_id = intval($_POST['product_id']);
        $enabled = sanitize_text_field($_POST['enabled']);
        $marketplace = isset($_POST['marketplace']) ? sanitize_text_field($_POST['marketplace']) : null;
        
        update_post_meta($product_id, '_tnt_enabled', $enabled);
        
        if ($marketplace !== null) {
            update_post_meta($product_id, '_tnt_marketplace', $marketplace);
        }
        
        // Sync if enabled (new enable or marketplace toggle on already-enabled product)
        if ($enabled === 'yes') {
            $sync_result = $this->sync_single_product($product_id);
            if (!$sync_result) {
                wp_send_json_error(['message' => 'Failed to sync product to Tap and Try. Please check your API key.']);
                return;
            }
        } else {
            // Disabled - unpublish from TapAndTry
            $tnt_product_id = get_post_meta($product_id, '_tnt_product_id', true);
            if (!empty($tnt_product_id)) {
                $this->api_request('/v1/plugin/products/' . $tnt_product_id, 'PUT', [
                    'status' => 'draft',
                    'marketplaceVisible' => false
                ]);
            }
        }
        
        wp_send_json_success(['message' => 'Product updated']);
    }
    
    /**
     * AJAX: Get Usage
     */
    public function ajax_get_usage() {
        check_ajax_referer('tnt_admin_nonce', 'nonce');
        
        $response = $this->api_request('/v1/plugin/usage');
        
        if ($response && isset($response['data'])) {
            wp_send_json_success($response['data']);
        } else {
            wp_send_json_error(['message' => 'Failed to fetch usage data']);
        }
    }
    
    /**
     * AJAX: Get Login URL for auto-login to Tap and Try dashboard
     */
    public function ajax_get_login_url() {
        check_ajax_referer('tnt_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }
        
        $redirect_to = isset($_POST['redirect_to']) ? sanitize_text_field($_POST['redirect_to']) : '/pricing';
        
        $response = $this->api_request('/v1/plugin/auth-token', 'POST', [
            'redirectTo' => $redirect_to
        ]);
        
        if ($response && isset($response['data']['loginUrl'])) {
            wp_send_json_success(['loginUrl' => $response['data']['loginUrl']]);
        } else {
            wp_send_json_error(['message' => 'Failed to generate login URL']);
        }
    }
    
    /**
     * Shortcode Handler
     */
    public function shortcode_handler($atts) {
        $atts = shortcode_atts([
            'product_id' => '',
            'slug' => '',
            'text' => get_option('tnt_button_text', 'Try It On'),
        ], $atts);
        
        $identifier = !empty($atts['slug']) ? $atts['slug'] : $atts['product_id'];
        
        if (empty($identifier)) {
            return '';
        }
        
        wp_enqueue_script('tnt-widget', TNT_WIDGET_URL, [], TNT_VERSION, true);
        
        return sprintf(
            '<div class="tnt-try-button" data-tnt-product="%s" data-tnt-text="%s"></div>',
            esc_attr($identifier),
            esc_attr($atts['text'])
        );
    }
    
    /**
     * Render Admin Pages
     */
    public function render_admin_page() {
        include TNT_PLUGIN_DIR . 'templates/admin-dashboard.php';
    }
    
    public function render_products_page() {
        include TNT_PLUGIN_DIR . 'templates/admin-products.php';
    }
    
    public function render_settings_page() {
        include TNT_PLUGIN_DIR . 'templates/admin-settings.php';
    }
}

// Initialize plugin
function tnt_init() {
    return TapAndTry::instance();
}
add_action('plugins_loaded', 'tnt_init');

// Activation hook
register_activation_hook(__FILE__, function() {
    // Register plugin installation with Tap and Try
    $api_key = get_option('tnt_api_key', '');
    if (!empty($api_key)) {
        wp_remote_post(TNT_API_URL . '/v1/plugin/register', [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode([
                'platform' => 'wordpress',
                'domain' => home_url(),
                'siteName' => get_bloginfo('name'),
                'platformVersion' => get_bloginfo('version'),
                'pluginVersion' => TNT_VERSION,
            ]),
        ]);
    }
});

// Deactivation hook
register_deactivation_hook(__FILE__, function() {
    // Optionally notify Tap and Try of deactivation
});
